/** @module obj */

function moduleFunction(config) {
  // Initialize $$ with provided config or default
  let $$ = config || require('./debug')({ debug: 'cache', path: module.id });

  /**
   * Validates if a string is a UUID
   * @param {string} uuid - The string to validate
   * @returns {boolean}
   */
  function isUUID(uuid) {
    let exp = '^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$';
    return new RegExp(exp).test(uuid);
  }

  /**
   * Check variable expression helper
   * @param {*} exp - variable expression
   * @returns {boolean}
   */
  function nullOrEmpty(exp) {
    return (
      typeof exp === 'undefined' ||
      exp === null ||
      (typeof exp === 'string' && exp.length === 0) ||
      (typeof exp === 'object' && Array.isArray(exp) && exp.length === 0) ||
      (typeof exp === 'object' && Object.keys(exp).length === 0)
    );
  }

  /**
   * Check if value is null or undefined
   * @param {*} exp - value to check
   * @returns {boolean}
   */
  function nullOrUndefined(exp) {
    return typeof exp === 'undefined' || exp === null;
  }

  /**
   * Finds the first occurrence of a key in an object or array recursively.
   *
   * @param {string} key - The key to search for.
   * @param {Object|Array} obj - The object or array to search in.
   * @returns {Object|null} - The object containing the key, or null if not found.
   */
  function findKey(key, obj) {
    if (!key || !obj || typeof obj !== 'object') return null;
    if (key in obj) return obj;
    if (Array.isArray(obj)) {
      for (let i = 0; i < obj.length; i++) {
        let result = findKey(key, obj[i]);
        if (result) return result;
      }
    } else {
      for (let k in obj) {
        let result = findKey(key, obj[k]);
        if (result) return result;
      }
    }
    return null;
  }

  /**
   * Finds and returns the first object in the given object hierarchy that has a property with the specified key and value.
   * @param {string|number} key - The key to search for.
   * @param {string|number} value - The value to match.
   * @param {object|array} obj - The object or array to search within.
   * @returns {object|null} - The first object that matches the key and value, or null if no match is found.
   */
  function findValue(key, value, obj) {
    if (!key || !obj || typeof obj !== 'object') return null;
    if (key in obj && obj[key].toString() === value.toString()) return obj;
    if (Array.isArray(obj)) {
      for (let i = 0; i < obj.length; i++) {
        let result = findValue(key, value, obj[i]);
        if (result) return result;
      }
    } else {
      for (let k in obj) {
        let result = findValue(key, value, obj[k]);
        if (result) return result;
      }
    }
    return null;
  }

  /**
   * Search and optionally update object based on search criteria
   * @param {Object} obj - Object to search in
   * @param {Object} searchCriteria - Search criteria
   * @param {Object} [assignObj] - Optional object to assign to found object
   * @returns {Object|null} - Found object or null
   */
  function search(obj, searchCriteria, assignObj) {
    let searchKey = Object.keys(searchCriteria)[0];
    let searchValue = searchCriteria[searchKey];

    function searchAndAssign(currentObj) {
      if (typeof currentObj !== 'object' || currentObj === null) {
        return null;
      }

      if (currentObj[searchKey] === searchValue) {
        if (assignObj) {
          Object.assign(currentObj, assignObj);
        }
        return currentObj;
      }

      for (let prop in currentObj) {
        if (Object.prototype.hasOwnProperty.call(currentObj, prop)) {
          if (Array.isArray(currentObj[prop])) {
            for (let i = 0; i < currentObj[prop].length; i++) {
              let result = searchAndAssign(currentObj[prop][i]);
              if (result) return result;
            }
          } else if (typeof currentObj[prop] === 'object') {
            let result = searchAndAssign(currentObj[prop]);
            if (result) return result;
          }
        }
      }
      return null;
    }

    return searchAndAssign(obj);
  }

  /**
   * Remove object based on search criteria
   * @param {Object} obj - Object to search in
   * @param {Object} searchCriteria - Search criteria
   * @returns {Object} - Modified object
   */
  function remove(obj, searchCriteria) {
    let searchKey = Object.keys(searchCriteria)[0];
    let searchValue = searchCriteria[searchKey];

    function searchAndRemove(currentObj, parentObj, parentKey) {
      if (typeof currentObj !== 'object' || currentObj === null) {
        return false;
      }

      if (currentObj[searchKey] === searchValue) {
        if (Array.isArray(parentObj)) {
          parentObj.splice(parentKey, 1);
        } else {
          delete parentObj[parentKey];
        }
        return true;
      }

      for (let prop in currentObj) {
        if (Object.prototype.hasOwnProperty.call(currentObj, prop)) {
          if (Array.isArray(currentObj[prop])) {
            for (let i = 0; i < currentObj[prop].length; i++) {
              if (searchAndRemove(currentObj[prop][i], currentObj[prop], i)) {
                return true;
              }
            }
          } else if (typeof currentObj[prop] === 'object') {
            if (searchAndRemove(currentObj[prop], currentObj, prop)) {
              return true;
            }
          }
        }
      }
      return false;
    }

    searchAndRemove(obj, null, null);
    return obj;
  }

  /**
   * Add or update object based on search criteria
   * @param {Object} obj - Object to search in
   * @param {Object} searchCriteria - Search criteria
   * @param {Object} newObj - Object to add
   * @param {boolean} [asArrayItem] - Add as array item flag
   * @returns {Object} - Modified object
   */
  function add(obj, searchCriteria, newObj, asArrayItem) {
    asArrayItem = asArrayItem || false;
    let searchKey = Object.keys(searchCriteria)[0];
    let searchValue = searchCriteria[searchKey];

    function searchAndAdd(currentObj) {
      if (typeof currentObj !== 'object' || currentObj === null) {
        return false;
      }

      if (currentObj[searchKey] === searchValue) {
        if (asArrayItem && Array.isArray(currentObj)) {
          currentObj.push(newObj);
        } else if (typeof newObj === 'object') {
          Object.assign(currentObj, newObj);
        }
        return true;
      }

      for (let prop in currentObj) {
        if (Object.prototype.hasOwnProperty.call(currentObj, prop)) {
          if (Array.isArray(currentObj[prop])) {
            for (let i = 0; i < currentObj[prop].length; i++) {
              if (searchAndAdd(currentObj[prop][i])) {
                return true;
              }
            }
          } else if (typeof currentObj[prop] === 'object') {
            if (searchAndAdd(currentObj[prop])) {
              return true;
            }
          }
        }
      }
      return false;
    }

    searchAndAdd(obj);
    return obj;
  }

  // Return the module functions
  return {
    isUUID: isUUID,
    nullOrEmpty: nullOrEmpty,
    nullOrUndefined: nullOrUndefined,
    findKey: findKey,
    findValue: findValue,
    search: search,
    update: search,
    remove: remove,
    add: add
  };
}

// Initialize $$ when the module is required without parameters
let moduleExports = moduleFunction();

// Export the module function and its methods
module.exports = Object.assign(moduleFunction, moduleExports);
